/*
 * EKF.c
 *
 *  Created on: Apr 7, 2015
 *      Author: Jordan
 */

#include "arm_math.h"
#include "EKF.h"

arm_matrix_instance_f32 AHRS_x, AHRS_u, AHRS_z, AHRS_x_, AHRS_f, AHRS_F, AHRS_h, AHRS_H, AHRS_Q, AHRS_R, AHRS_P, AHRS_P_, AHRS_y, AHRS_S, AHRS_K;

arm_matrix_instance_f32 INS_x, INS_u, INS_z, INS_x_, INS_f, INS_F, INS_h, INS_H, INS_Q, INS_R, INS_P, INS_P_, INS_y, INS_S, INS_K;

float32_t AHRS_states[AHRS_N_STATES];
float32_t AHRS_u_data[AHRS_N_INPUTS];
float32_t AHRS_z_data[AHRS_N_MEAS];
float32_t AHRS_x__data[AHRS_N_STATES];
float32_t AHRS_f_data[AHRS_N_STATES];
float32_t AHRS_F_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_h_data[AHRS_N_MEAS];
float32_t AHRS_H_data[AHRS_N_MEAS*AHRS_N_STATES];
float32_t AHRS_Q_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_R_data[AHRS_N_MEAS*AHRS_N_MEAS];
float32_t AHRS_P_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_P__data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_y_data[AHRS_N_MEAS];
float32_t AHRS_S_data[AHRS_N_MEAS*AHRS_N_MEAS];
float32_t AHRS_K_data[AHRS_N_STATES*AHRS_N_MEAS];

float32_t INS_states[INS_N_STATES];
float32_t INS_u_data[INS_N_INPUTS];
float32_t INS_z_data[INS_N_MEAS];
float32_t INS_x__data[INS_N_STATES];
float32_t INS_f_data[INS_N_STATES];
float32_t INS_F_data[INS_N_STATES*INS_N_STATES];
float32_t INS_h_data[INS_N_MEAS];
float32_t INS_H_data[INS_N_MEAS*INS_N_STATES];
float32_t INS_Q_data[INS_N_STATES*INS_N_STATES];
float32_t INS_R_data[INS_N_MEAS*INS_N_MEAS];
float32_t INS_P_data[INS_N_STATES*INS_N_STATES];
float32_t INS_P__data[INS_N_STATES*INS_N_STATES];
float32_t INS_y_data[INS_N_MEAS];
float32_t INS_S_data[INS_N_MEAS*INS_N_MEAS];
float32_t INS_K_data[INS_N_STATES*INS_N_MEAS];

arm_matrix_instance_f32 AHRS_Ft, AHRS_P_Ft, AHRS_Ht, AHRS_Sinv, AHRS_P_Ht, AHRS_I, AHRS_K_H;

arm_matrix_instance_f32 INS_Ft, INS_P_Ft, INS_Ht, INS_Sinv, INS_P_Ht, INS_I, INS_K_H;

float32_t AHRS_Ft_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_P_Ft_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_Ht_data[AHRS_N_STATES*AHRS_N_MEAS];
float32_t AHRS_Sinv_data[AHRS_N_MEAS*AHRS_N_MEAS];
float32_t AHRS_P_Ht_data[AHRS_N_STATES*AHRS_N_MEAS];
float32_t AHRS_I_data[AHRS_N_STATES*AHRS_N_STATES];
float32_t AHRS_K_H_data[AHRS_N_STATES*AHRS_N_STATES];

float32_t INS_Ft_data[INS_N_STATES*INS_N_STATES];
float32_t INS_P_Ft_data[INS_N_STATES*INS_N_STATES];
float32_t INS_Ht_data[INS_N_STATES*INS_N_MEAS];
float32_t INS_Sinv_data[INS_N_MEAS*INS_N_MEAS];
float32_t INS_P_Ht_data[INS_N_STATES*INS_N_MEAS];
float32_t INS_I_data[INS_N_STATES*INS_N_STATES];
float32_t INS_K_H_data[INS_N_STATES*INS_N_STATES];

void update_AHRS_u(float32_t wx, float32_t wy, float32_t wz){
	AHRS_u_data[0] = wx;
	AHRS_u_data[1] = wy;
	AHRS_u_data[2] = wz;
}

void update_INS_u(float32_t ax, float32_t ay, float32_t az){
	INS_u_data[0] = ax;
	INS_u_data[1] = ay;
	INS_u_data[2] = az;
}

void update_AHRS_z(float32_t ax, float32_t ay, float32_t az, float32_t mag_x, float32_t mag_y, float32_t mag_z){
	float32_t acc_norm = sqrtf(powf(ax,2)+powf(ay,2)+powf(az,2));
	float32_t sc = fabs(acc_norm - 1)*100 + 1;
	AHRS_R_data[0] = A_VAR*sc;
	AHRS_R_data[7] = A_VAR*sc;
	AHRS_R_data[14] = A_VAR*sc;
	AHRS_z_data[0] = ax;
	AHRS_z_data[1] = ay;
	AHRS_z_data[2] = az;

	float32_t mag_norm = sqrtf(powf(mag_x,2)+powf(mag_y,2)+powf(mag_z,2));
	AHRS_z_data[3] = mag_x/mag_norm;
	AHRS_z_data[4] = mag_y/mag_norm;
	AHRS_z_data[5] = mag_z/mag_norm;
}

void update_INS_z(float32_t px, float32_t py, float32_t pz, float32_t vx, float32_t vy){
	INS_z_data[0] = px;
	INS_z_data[1] = py;
	INS_z_data[2] = pz;
	INS_z_data[3] = vx;
	INS_z_data[4] = vy;
}

void init_AHRS(float32_t qs, float32_t qx, float32_t qy, float32_t qz, float32_t bwx, float32_t bwy, float32_t bwz){
	AHRS_states[0] = qs;
	AHRS_states[1] = qx;
	AHRS_states[2] = qy;
	AHRS_states[3] = qz;
	AHRS_states[4] = bwx;
	AHRS_states[5] = bwy;
	AHRS_states[6] = bwz;

	float32_t P_tmp[AHRS_N_STATES*AHRS_N_STATES] = {
		A_VAR,	0,		0,		0,		0,		0,		0,
		0,		A_VAR,	0,		0,		0,		0,		0,
		0,		0,		A_VAR,	0,		0,		0,		0,
		0,		0,		0,		A_VAR,	0,		0,		0,
		0,		0,		0,		0,		0,		0,		0,
		0,		0,		0,		0,		0,		0,		0,
		0,		0,		0,		0,		0,		0,		0,
	};
	memcpy(AHRS_P_data, P_tmp, AHRS_N_STATES*AHRS_N_STATES*sizeof(float32_t));

	float32_t Q_tmp[AHRS_N_STATES*AHRS_N_STATES] = {
		Q_VAR,	0,		0,		0,		0,		0,		0,
		0,		Q_VAR,	0,		0,		0,		0,		0,
		0,		0,		Q_VAR,	0,		0,		0,		0,
		0,		0,		0,		Q_VAR,	0,		0,		0,
		0,		0,		0,		0,		BW_VAR,	0,		0,
		0,		0,		0,		0,		0,		BW_VAR,	0,
		0,		0,		0,		0,		0,		0,		BW_VAR,
	};
	memcpy(AHRS_Q_data, Q_tmp, AHRS_N_STATES*AHRS_N_STATES*sizeof(float32_t));

	float32_t R_tmp[AHRS_N_MEAS*AHRS_N_MEAS] = {
		A_VAR,	0,		0,		0,		0,		0,
		0,		A_VAR,	0,		0,		0,		0,
		0,		0,		A_VAR,	0,		0,		0,
		0,		0,		0,		M_VAR,	0,		0,
		0,		0,		0,		0,		M_VAR,	0,
		0,		0,		0,		0,		0,		M_VAR,
	};
	memcpy(AHRS_R_data, R_tmp, AHRS_N_MEAS*AHRS_N_MEAS*sizeof(float32_t));

	float32_t I_tmp[AHRS_N_STATES*AHRS_N_STATES] = {
		1,		0,		0,		0,		0,		0,		0,
		0,		1,		0,		0,		0,		0,		0,
		0,		0,		1,		0,		0,		0,		0,
		0,		0,		0,		1,		0,		0,		0,
		0,		0,		0,		0,		1,		0,		0,
		0,		0,		0,		0,		0,		1,		0,
		0,		0,		0,		0,		0,		0,		1,
	};
	memcpy(AHRS_I_data, I_tmp, AHRS_N_STATES*AHRS_N_STATES*sizeof(float32_t));

	AHRS_init_matrices();
}

void init_INS(float32_t px, float32_t py, float32_t pz, float32_t vx, float32_t vy, float32_t vz, float32_t bax, float32_t bay, float32_t baz){
	INS_states[0] = px;
	INS_states[1] = py;
	INS_states[2] = pz;
	INS_states[3] = vx;
	INS_states[4] = vy;
	INS_states[5] = vz;
	INS_states[6] = bax;
	INS_states[7] = bay;
	INS_states[8] = baz;

	float32_t P_tmp[INS_N_STATES*INS_N_STATES] = {
		P_GPS_VAR,	0,			0,			0,			0,			0,			0,			0,			0,
		0,			P_GPS_VAR,	0,			0,			0,			0,			0,			0,			0,
		0,			0,			P_ALT_VAR,	0,			0,			0,			0,			0,			0,
		0,			0,			0,			V_GPS_VAR,	0,			0,			0,			0,			0,
		0,			0,			0,			0,			V_GPS_VAR,	0,			0,			0,			0,
		0,			0,			0,			0,			0,			P_ALT_VAR,	0,			0,			0,
		0,			0,			0,			0,			0,			0,			0,			0,			0,
		0,			0,			0,			0,			0,			0,			0,			0,			0,
		0,			0,			0,			0,			0,			0,			0,			0,			0,
	};
	memcpy(INS_P_data, P_tmp, INS_N_STATES*INS_N_STATES*sizeof(float32_t));

	float32_t Q_tmp[INS_N_STATES*INS_N_STATES] = {
		P_VAR,	0,		0,		0,		0,		0,		0,		0,		0,
		0,		P_VAR,	0,		0,		0,		0,		0,		0,		0,
		0,		0,		P_VAR,	0,		0,		0,		0,		0,		0,
		0,		0,		0,		V_VAR,	0,		0,		0,		0,		0,
		0,		0,		0,		0,		V_VAR,	0,		0,		0,		0,
		0,		0,		0,		0,		0,		VZ_VAR,	0,		0,		0,
		0,		0,		0,		0,		0,		0,		BA_VAR,	0,		0,
		0,		0,		0,		0,		0,		0,		0,		BA_VAR,	0,
		0,		0,		0,		0,		0,		0,		0,		0,		BA_VAR,
	};
	memcpy(INS_Q_data, Q_tmp, INS_N_STATES*INS_N_STATES*sizeof(float32_t));

	float32_t R_tmp[INS_N_MEAS*INS_N_MEAS] = {
		P_GPS_VAR,	0,			0,			0,			0,
		0,			P_GPS_VAR,	0,			0,			0,
		0,			0,			P_ALT_VAR,	0,			0,
		0,			0,			0,			V_GPS_VAR,	0,
		0,			0,			0,			0,			V_GPS_VAR,
	};
	memcpy(INS_R_data, R_tmp, INS_N_MEAS*INS_N_MEAS*sizeof(float32_t));

	float32_t I_tmp[INS_N_STATES*INS_N_STATES] = {
		1,		0,		0,		0,		0,		0,		0,		0,		0,
		0,		1,		0,		0,		0,		0,		0,		0,		0,
		0,		0,		1,		0,		0,		0,		0,		0,		0,
		0,		0,		0,		1,		0,		0,		0,		0,		0,
		0,		0,		0,		0,		1,		0,		0,		0,		0,
		0,		0,		0,		0,		0,		1,		0,		0,		0,
		0,		0,		0,		0,		0,		0,		1,		0,		0,
		0,		0,		0,		0,		0,		0,		0,		1,		0,
		0,		0,		0,		0,		0,		0,		0,		0,		1,
	};
	memcpy(INS_I_data, I_tmp, INS_N_STATES*INS_N_STATES*sizeof(float32_t));

	INS_init_matrices();
}

void AHRS_calc_f(){
	float32_t qs = AHRS_states[0];
	float32_t qx = AHRS_states[1];
	float32_t qy = AHRS_states[2];
	float32_t qz = AHRS_states[3];
	float32_t bwx = AHRS_states[4];
	float32_t bwy = AHRS_states[5];
	float32_t bwz = AHRS_states[6];

	float32_t wx = AHRS_u_data[0];
	float32_t wy = AHRS_u_data[1];
	float32_t wz = AHRS_u_data[2];

	AHRS_f_data[0] = (qx*(bwx - wx))/2 + (qy*(bwy - wy))/2 + (qz*(bwz - wz))/2;
	AHRS_f_data[1] = (qz*(bwy - wy))/2 - (qs*(bwx - wx))/2 - (qy*(bwz - wz))/2;
	AHRS_f_data[2] = (qx*(bwz - wz))/2 - (qz*(bwx - wx))/2 - (qs*(bwy - wy))/2;
	AHRS_f_data[3] = (qy*(bwx - wx))/2 - (qs*(bwz - wz))/2 - (qx*(bwy - wy))/2;
	AHRS_f_data[4] = 0;
	AHRS_f_data[5] = 0;
	AHRS_f_data[6] = 0;
}

void INS_calc_f(){
	float32_t qs = AHRS_states[0];
	float32_t qx = AHRS_states[1];
	float32_t qy = AHRS_states[2];
	float32_t qz = AHRS_states[3];

	float32_t vx = INS_states[3];
	float32_t vy = INS_states[4];
	float32_t vz = INS_states[5];
	float32_t bax = INS_states[6];
	float32_t bay = INS_states[7];
	float32_t baz = INS_states[8];

	float32_t ax = INS_u_data[0];
	float32_t ay = INS_u_data[1];
	float32_t az = INS_u_data[2];

	INS_f_data[0] = vx;
	INS_f_data[1] = vy;
	INS_f_data[2] = vz;
	INS_f_data[3] = (az - baz)*(2*qs*qy + 2*qx*qz) - (ay - bay)*(2*qs*qz - 2*qx*qy) - (ax - bax)*(2*powf(qy,2) + 2*powf(qz,2) - 1);
	INS_f_data[4] = (ax - bax)*(2*qs*qz + 2*qx*qy) - (ay - bay)*(2*powf(qx,2) + 2*powf(qz,2) - 1) - (az - baz)*(2*qs*qx - 2*qy*qz);
	INS_f_data[5] = (ay - bay)*(2*qs*qx + 2*qy*qz) - (ax - bax)*(2*qs*qy - 2*qx*qz) - (az - baz)*(2*powf(qx,2) + 2*powf(qy,2) - 1) + 9.81;
	INS_f_data[6] = 0;
	INS_f_data[7] = 0;
	INS_f_data[8] = 0;
}

void AHRS_calc_F(){
	float32_t qs = AHRS_states[0];
	float32_t qx = AHRS_states[1];
	float32_t qy = AHRS_states[2];
	float32_t qz = AHRS_states[3];
	float32_t bwx = AHRS_states[4];
	float32_t bwy = AHRS_states[5];
	float32_t bwz = AHRS_states[6];

	float32_t wx = AHRS_u_data[0];
	float32_t wy = AHRS_u_data[1];
	float32_t wz = AHRS_u_data[2];

	float32_t F_tmp[AHRS_N_STATES*AHRS_N_STATES] = {
		             0, bwx/2 - wx/2, bwy/2 - wy/2, bwz/2 - wz/2,  qx/2,  qy/2,  qz/2,
		  wx/2 - bwx/2,            0, wz/2 - bwz/2, bwy/2 - wy/2, -qs/2,  qz/2, -qy/2,
		  wy/2 - bwy/2, bwz/2 - wz/2,            0, wx/2 - bwx/2, -qz/2, -qs/2,  qx/2,
		  wz/2 - bwz/2, wy/2 - bwy/2, bwx/2 - wx/2,            0,  qy/2, -qx/2, -qs/2,
		             0,            0,            0,            0,     0,     0,     0,
		             0,            0,            0,            0,     0,     0,     0,
		             0,            0,            0,            0,     0,     0,     0,
	};
	memcpy(AHRS_F_data, F_tmp, AHRS_N_STATES*AHRS_N_STATES*sizeof(float32_t));
}

void INS_calc_F(){
	float32_t qs = AHRS_states[0];
	float32_t qx = AHRS_states[1];
	float32_t qy = AHRS_states[2];
	float32_t qz = AHRS_states[3];

	float32_t F_tmp[INS_N_STATES*INS_N_STATES] = {
		0,		0,		0,		1,		0,		0,		0,									0,									0,
		0,		0,		0,		0,		1,		0,		0,									0,									0,
		0,		0,		0,		0,		0,		1,		0,									0,									0,
		0,		0,		0,		0,		0,		0,		2*powf(qy,2) + 2*powf(qz,2) - 1,	2*qs*qz - 2*qx*qy,					-2*qs*qy - 2*qx*qz,
		0,		0,		0,		0,		0,		0,		-2*qs*qz - 2*qx*qy,					2*powf(qx,2) + 2*powf(qz,2) - 1,	2*qs*qx - 2*qy*qz,
		0,		0,		0,		0,		0,		0,		2*qs*qy - 2*qx*qz,					-2*qs*qx - 2*qy*qz,					2*powf(qx,2) + 2*powf(qy,2) - 1,
		0,		0,		0,		0,		0,		0,		0,									0,									0,
		0,		0,		0,		0,		0,		0,		0,									0,									0,
		0,		0,		0,		0,		0,		0,		0,									0,									0,
	};
	memcpy(INS_F_data, F_tmp, INS_N_STATES*INS_N_STATES*sizeof(float32_t));
}

void AHRS_calc_h(){
	float32_t qs = AHRS_x__data[0];
	float32_t qx = AHRS_x__data[1];
	float32_t qy = AHRS_x__data[2];
	float32_t qz = AHRS_x__data[3];

	AHRS_h_data[0] = 2*qx*qz - 2*qs*qy;
	AHRS_h_data[1] = 2*qs*qx + 2*qy*qz;
	AHRS_h_data[2] = -2*powf(qx,2) - 2*powf(qy,2) + 1;
	AHRS_h_data[3] = EARTH_BY*(2*qs*qz + 2*qx*qy) - EARTH_BX*(2*powf(qy,2) + 2*powf(qz,2) - 1) - EARTH_BZ*(2*qs*qy - 2*qx*qz);
	AHRS_h_data[4] = EARTH_BZ*(2*qs*qx + 2*qy*qz) - EARTH_BX*(2*qs*qz - 2*qx*qy) - EARTH_BY*(2*powf(qx,2) + 2*powf(qz,2) - 1);
	AHRS_h_data[5] = EARTH_BX*(2*qs*qy + 2*qx*qz) - EARTH_BZ*(2*powf(qx,2) + 2*powf(qy,2) - 1) - EARTH_BY*(2*qs*qx - 2*qy*qz);
}

void INS_calc_h(){
	float32_t px = INS_x__data[0];
	float32_t py = INS_x__data[1];
	float32_t pz = INS_x__data[2];
	float32_t vx = INS_x__data[3];
	float32_t vy = INS_x__data[4];

	INS_h_data[0] = px;
	INS_h_data[1] = py;
	INS_h_data[2] = pz;
	INS_h_data[3] = vx;
	INS_h_data[4] = vy;
}

void AHRS_calc_H(){
	float32_t qs = AHRS_x__data[0];
	float32_t qx = AHRS_x__data[1];
	float32_t qy = AHRS_x__data[2];
	float32_t qz = AHRS_x__data[3];

	float32_t H_tmp[AHRS_N_MEAS*AHRS_N_STATES] = {
		  -2*qy,  							2*qz, 											-2*qs, 											2*qx, 											0, 		0, 		0,
		  2*qx,  							2*qs,  											2*qz, 											2*qy, 											0, 		0, 		0,
		  0, 								-4*qx, 											-4*qy,    										0, 												0, 		0, 		0,
		  2*EARTH_BY*qz - 2*EARTH_BZ*qy,	2*EARTH_BY*qy + 2*EARTH_BZ*qz,					2*EARTH_BY*qx - 4*EARTH_BX*qy - 2*EARTH_BZ*qs,	2*EARTH_BY*qs - 4*EARTH_BX*qz + 2*EARTH_BZ*qx,	0,		0,		0,
		  2*EARTH_BZ*qx - 2*EARTH_BX*qz,	2*EARTH_BZ*qs + 2*EARTH_BX*qy - 4*EARTH_BY*qx,	2*EARTH_BX*qx + 2*EARTH_BZ*qz,					2*EARTH_BZ*qy - 4*EARTH_BY*qz - 2*EARTH_BX*qs,	0,		0,		0,
		  2*EARTH_BX*qy - 2*EARTH_BY*qx,	2*EARTH_BX*qz - 2*EARTH_BY*qs - 4*EARTH_BZ*qx,	2*EARTH_BX*qs + 2*EARTH_BY*qz - 4*EARTH_BZ*qy,	2*EARTH_BX*qx + 2*EARTH_BY*qy,					0,		0,		0,
	};
	memcpy(AHRS_H_data, H_tmp, AHRS_N_MEAS*AHRS_N_STATES*sizeof(float32_t));
}

void INS_calc_H(){
	float32_t H_tmp[INS_N_MEAS*INS_N_STATES] = {
		1,	0,	0,	0,	0,	0,	0,	0,	0,
		0,	1,	0,	0,	0,	0,	0,	0,	0,
		0,	0,	1,	0,	0,	0,	0,	0,	0,
		0,	0,	0,	1,	0,	0,	0,	0,	0,
		0,	0,	0,	0,	1,	0,	0,	0,	0,
	};
	memcpy(INS_H_data, H_tmp, INS_N_MEAS*INS_N_STATES*sizeof(float32_t));
}

void AHRS_init_matrices(){
	arm_mat_init_f32(&AHRS_x, AHRS_N_STATES, 1, AHRS_states);
	arm_mat_init_f32(&AHRS_u, AHRS_N_INPUTS, 1, AHRS_u_data);
	arm_mat_init_f32(&AHRS_z, AHRS_N_MEAS, 1, AHRS_z_data);
	arm_mat_init_f32(&AHRS_x_, AHRS_N_STATES, 1, AHRS_x__data);
	arm_mat_init_f32(&AHRS_f, AHRS_N_STATES, 1, AHRS_f_data);
	arm_mat_init_f32(&AHRS_F, AHRS_N_STATES, AHRS_N_STATES, AHRS_F_data);
	arm_mat_init_f32(&AHRS_h, AHRS_N_MEAS, 1, AHRS_h_data);
	arm_mat_init_f32(&AHRS_H, AHRS_N_MEAS, AHRS_N_STATES, AHRS_H_data);
	arm_mat_init_f32(&AHRS_Q, AHRS_N_STATES, AHRS_N_STATES, AHRS_Q_data);
	arm_mat_init_f32(&AHRS_R, AHRS_N_MEAS, AHRS_N_MEAS, AHRS_R_data);
	arm_mat_init_f32(&AHRS_P, AHRS_N_STATES, AHRS_N_STATES, AHRS_P_data);
	arm_mat_init_f32(&AHRS_P_, AHRS_N_STATES, AHRS_N_STATES, AHRS_P__data);
	arm_mat_init_f32(&AHRS_y, AHRS_N_MEAS, 1, AHRS_y_data);
	arm_mat_init_f32(&AHRS_S, AHRS_N_MEAS, AHRS_N_MEAS, AHRS_S_data);
	arm_mat_init_f32(&AHRS_K, AHRS_N_STATES, AHRS_N_MEAS, AHRS_K_data);

	arm_mat_init_f32(&AHRS_Ft, AHRS_N_STATES, AHRS_N_STATES, AHRS_Ft_data);
	arm_mat_init_f32(&AHRS_P_Ft, AHRS_N_STATES, AHRS_N_STATES, AHRS_P_Ft_data);
	arm_mat_init_f32(&AHRS_Ht, AHRS_N_STATES, AHRS_N_MEAS, AHRS_Ht_data);
	arm_mat_init_f32(&AHRS_Sinv, AHRS_N_MEAS, AHRS_N_MEAS, AHRS_Sinv_data);
	arm_mat_init_f32(&AHRS_P_Ht, AHRS_N_STATES, AHRS_N_MEAS, AHRS_P_Ht_data);
	arm_mat_init_f32(&AHRS_I, AHRS_N_STATES, AHRS_N_STATES, AHRS_I_data);
	arm_mat_init_f32(&AHRS_K_H, AHRS_N_STATES, AHRS_N_STATES, AHRS_K_H_data);
}

void INS_init_matrices(){
	arm_mat_init_f32(&INS_x, INS_N_STATES, 1, INS_states);
	arm_mat_init_f32(&INS_u, INS_N_INPUTS, 1, INS_u_data);
	arm_mat_init_f32(&INS_z, INS_N_MEAS, 1, INS_z_data);
	arm_mat_init_f32(&INS_x_, INS_N_STATES, 1, INS_x__data);
	arm_mat_init_f32(&INS_f, INS_N_STATES, 1, INS_f_data);
	arm_mat_init_f32(&INS_F, INS_N_STATES, INS_N_STATES, INS_F_data);
	arm_mat_init_f32(&INS_h, INS_N_MEAS, 1, INS_h_data);
	arm_mat_init_f32(&INS_H, INS_N_MEAS, INS_N_STATES, INS_H_data);
	arm_mat_init_f32(&INS_Q, INS_N_STATES, INS_N_STATES, INS_Q_data);
	arm_mat_init_f32(&INS_R, INS_N_MEAS, INS_N_MEAS, INS_R_data);
	arm_mat_init_f32(&INS_P, INS_N_STATES, INS_N_STATES, INS_P_data);
	arm_mat_init_f32(&INS_P_, INS_N_STATES, INS_N_STATES, INS_P__data);
	arm_mat_init_f32(&INS_y, INS_N_MEAS, 1, INS_y_data);
	arm_mat_init_f32(&INS_S, INS_N_MEAS, INS_N_MEAS, INS_S_data);
	arm_mat_init_f32(&INS_K, INS_N_STATES, INS_N_MEAS, INS_K_data);

	arm_mat_init_f32(&INS_Ft, INS_N_STATES, INS_N_STATES, INS_Ft_data);
	arm_mat_init_f32(&INS_P_Ft, INS_N_STATES, INS_N_STATES, INS_P_Ft_data);
	arm_mat_init_f32(&INS_Ht, INS_N_STATES, INS_N_MEAS, INS_Ht_data);
	arm_mat_init_f32(&INS_Sinv, INS_N_MEAS, INS_N_MEAS, INS_Sinv_data);
	arm_mat_init_f32(&INS_P_Ht, INS_N_STATES, INS_N_MEAS, INS_P_Ht_data);
	arm_mat_init_f32(&INS_I, INS_N_STATES, INS_N_STATES, INS_I_data);
	arm_mat_init_f32(&INS_K_H, INS_N_STATES, INS_N_STATES, INS_K_H_data);
}

void AHRS_step(){
	AHRS_calc_f();
	AHRS_calc_F();

	// x_hat_ = x_hat + 1/fs*f(x,u)
	arm_mat_scale_f32(&AHRS_f, 1.0/FS, &AHRS_f);
	arm_mat_add_f32(&AHRS_x, &AHRS_f, &AHRS_x_);

	// P_ = P + 1/fs*(F(x,u)*P + P*F(x,u)^T + Q)
	arm_mat_trans_f32(&AHRS_F, &AHRS_Ft);
	arm_mat_mult_f32(&AHRS_P, &AHRS_Ft, &AHRS_P_Ft);
	arm_mat_mult_f32(&AHRS_F, &AHRS_P, &AHRS_P_);
	arm_mat_add_f32(&AHRS_P_, &AHRS_P_Ft, &AHRS_P_);
	arm_mat_add_f32(&AHRS_P_, &AHRS_Q, &AHRS_P_);
	arm_mat_scale_f32(&AHRS_P_, 1.0/FS, &AHRS_P_);
	arm_mat_add_f32(&AHRS_P_, &AHRS_P, &AHRS_P_);

	AHRS_calc_h();
	AHRS_calc_H();

	// y = z - h(x)
	arm_mat_sub_f32(&AHRS_z, &AHRS_h, &AHRS_y);

	// S = H(x)*P_*H(x)^T + R
	arm_mat_trans_f32(&AHRS_H, &AHRS_Ht);
	arm_mat_mult_f32(&AHRS_P_, &AHRS_Ht, &AHRS_P_Ht);
	arm_mat_mult_f32(&AHRS_H, &AHRS_P_Ht, &AHRS_S);
	arm_mat_add_f32(&AHRS_S, &AHRS_R, &AHRS_S);

	// K = P_*H(x)^T*S^-1
	arm_mat_inverse_f32(&AHRS_S, &AHRS_Sinv);
	arm_mat_mult_f32(&AHRS_P_Ht, &AHRS_Sinv, &AHRS_K);

	// x_hat = x_hat_ + K*y;
	arm_mat_mult_f32(&AHRS_K, &AHRS_y, &AHRS_x);
	arm_mat_add_f32(&AHRS_x_, &AHRS_x, &AHRS_x);

	// P = (I-K*H(x))*P_
	arm_mat_mult_f32(&AHRS_K, &AHRS_H, &AHRS_K_H);
	arm_mat_sub_f32(&AHRS_I, &AHRS_K_H, &AHRS_K_H);
	arm_mat_mult_f32(&AHRS_K_H, &AHRS_P_, &AHRS_P);
}

void INS_step(){
	INS_calc_f();
	INS_calc_F();

	// x_hat_ = x_hat + 1/fs*f(x,u)
	arm_mat_scale_f32(&INS_f, 1.0/FS, &INS_f);
	arm_mat_add_f32(&INS_x, &INS_f, &INS_x_);

	// P_ = P + 1/fs*(F(x,u)*P + P*F(x,u)^T + Q)
	arm_mat_trans_f32(&INS_F, &INS_Ft);
	arm_mat_mult_f32(&INS_P, &INS_Ft, &INS_P_Ft);
	arm_mat_mult_f32(&INS_F, &INS_P, &INS_P_);
	arm_mat_add_f32(&INS_P_, &INS_P_Ft, &INS_P_);
	arm_mat_add_f32(&INS_P_, &INS_Q, &INS_P_);
	arm_mat_scale_f32(&INS_P_, 1.0/FS, &INS_P_);
	arm_mat_add_f32(&INS_P_, &INS_P, &INS_P_);

	INS_calc_h();
	INS_calc_H();

	// y = z - h(x)
	arm_mat_sub_f32(&INS_z, &INS_h, &INS_y);

	// S = H(x)*P_*H(x)^T + R
	arm_mat_trans_f32(&INS_H, &INS_Ht);
	arm_mat_mult_f32(&INS_P_, &INS_Ht, &INS_P_Ht);
	arm_mat_mult_f32(&INS_H, &INS_P_Ht, &INS_S);
	arm_mat_add_f32(&INS_S, &INS_R, &INS_S);

	// K = P_*H(x)^T*S^-1
	arm_mat_inverse_f32(&INS_S, &INS_Sinv);
	arm_mat_mult_f32(&INS_P_Ht, &INS_Sinv, &INS_K);

	// x_hat = x_hat_ + K*y;
	arm_mat_mult_f32(&INS_K, &INS_y, &INS_x);
	arm_mat_add_f32(&INS_x_, &INS_x, &INS_x);

	// P = (I-K*H(x))*P_
	arm_mat_mult_f32(&INS_K, &INS_H, &INS_K_H);
	arm_mat_sub_f32(&INS_I, &INS_K_H, &INS_K_H);
	arm_mat_mult_f32(&INS_K_H, &INS_P_, &INS_P);
}

void AHRS_normalize_q(){
	float32_t quat_norm = sqrtf(powf(AHRS_states[0],2)+powf(AHRS_states[1],2)+powf(AHRS_states[2],2)+powf(AHRS_states[3],2));

	AHRS_states[0] /= quat_norm;
	AHRS_states[1] /= quat_norm;
	AHRS_states[2] /= quat_norm;
	AHRS_states[3] /= quat_norm;
}
